#include "DMMessageReceiver.h"

DMMessageReceiver::DMMessageReceiver( const std::string& bURI,
				      const std::string& dName,
				      bool uTopic,
				      bool sTransacted) : connection(NULL),
							  session(NULL),
							  destination(NULL),
							  consumer(NULL),
							  brokerURI(bURI),
							  destinationName(dName),
							  useTopic(uTopic),
							  sessionTransacted(sTransacted) {}



void DMMessageReceiver::run()
{
    DMMessageDecoder decoder();
    this->messageCount = 0;
    
    try {
	printf("Receiver is running\n");
	// Create a ConnectionFactory
	auto_ptr<ConnectionFactory> connectionFactory(ConnectionFactory::createCMSConnectionFactory( brokerURI ) );
    
	// Create a Connection
	connection = connectionFactory->createConnection();
	connection->start();
	connection->setExceptionListener(this);
	
	// Create a Session
	if( sessionTransacted ) {
	    session = connection->createSession( Session::SESSION_TRANSACTED );
	} else {
	    session = connection->createSession( Session::AUTO_ACKNOWLEDGE );
	}
	
	// Create the destination (Topic or Queue)
	if( useTopic ) {
	    destination = session->createTopic( destinationName );
	} else {
	    destination = session->createQueue( destinationName );
	}
	
	// Create a MessageConsumer from the Session to the Topic or Queue
	consumer = session->createConsumer( destination );
	
	consumer->setMessageListener( this );

	
    } catch( CMSException& e ) {
	e.printStackTrace();
    }
}

DMMessageReceiver::~DMMessageReceiver(){
    this->cleanup();
}

void DMMessageReceiver::close() {
    this->cleanup();
}

// Called from the consumer since this class is a registered MessageListener.
void DMMessageReceiver::onMessage( const Message* message ){
    try
    {
	const TextMessage* textMessage = dynamic_cast< const TextMessage* >( message );
	string text = "";
	
	if( textMessage != NULL ) {
	    text = textMessage->getText();
	} else {
	    text = "";
	}
	
	decoder.decode(text);
    } catch (CMSException& e) {
	e.printStackTrace();
    }
    
    // Commit all messages.
    if( sessionTransacted ) {
	session->commit();
    }
}

// If something bad happens you see it here as this class is also been
// registered as an ExceptionListener with the connection.
void DMMessageReceiver::onException( const CMSException& ex AMQCPP_UNUSED) {
    printf("CMS Exception occurred.  Shutting down client.\n");
    ex.printStackTrace();
    exit(1);
}

void DMMessageReceiver::transportInterrupted() 
{
    std::cout << "The DMReceiver transport has been Interrupted." << std::endl;
}

void DMMessageReceiver::transportResumed()
{
    std::cout << "The DMReceiver transport has been Restored." << std::endl;
}


void DMMessageReceiver::cleanup(){

    //*************************************************
    // Always close destination, consumers and producers before
    // you destroy their sessions and connection.
    //*************************************************
    
    // Destroy resources.
    try{
	if( destination != NULL ) delete destination;
    }catch (CMSException& e) { e.printStackTrace(); }
    destination = NULL;
    
    try{
	if( consumer != NULL ) delete consumer;
    }catch (CMSException& e) { e.printStackTrace(); }
    consumer = NULL;
    
    // Close open resources.
    try{
	if( session != NULL ) session->close();
	if( connection != NULL ) connection->close();
    }catch (CMSException& e) { e.printStackTrace(); }
    
    // Now Destroy them
    try{
	if( session != NULL ) delete session;
    }catch (CMSException& e) { e.printStackTrace(); }
    session = NULL;
    
    try{
	if( connection != NULL ) delete connection;
    }catch (CMSException& e) { e.printStackTrace(); }
    connection = NULL;
}


